#include "gba.h"
#include "sound.h"
#include <malloc.h>

#define	INT_TM0		(0x0008)
#define	INT_TM1		(0x0010)

#include "laser2.h"
#include "exp2.h"

void play_sfx(const s8* data, int length);

SAMPLE laser_sample = {laser_wav, 4032};
SAMPLE exp_sample = {exp_wav, 16128};

/* mixing buffer for sound system */
s8* soundbuffers[2];
u32 sbuffer = 0;

#define SOUNDBUFFER_LENGTH 400

typedef struct
{
	const s8* data;
	u32 length, cur;
} buffer_sound;

#define SOUNDBUFFER_CHANNELS 8
buffer_sound sounds[SOUNDBUFFER_CHANNELS];

// init_sfx_system - sets the registers for enabling the sound hardware and
//                   Direct Sound
// PARAMETERS:  none
// RETURNS:     none
void init_sfx_system(void)
{
	// turn on the sound chip
	REG_SOUNDCNT_X = SND_ENABLED;

	// make sure sound channels 1-4 are turned off
	REG_SOUNDCNT_L = 0;

	// set the direct sound output control register
	REG_SOUNDCNT_H = SND_OUTPUT_RATIO_100 | // 100% sound output
		DSA_OUTPUT_RATIO_100 | // 100% direct sound A output
		DSA_OUTPUT_TO_BOTH |   // output Direct Sound A to both right and left speakers
		DSA_TIMER0 |           // use timer 0 to determine the playback frequency of Direct Sound A
		DSA_FIFO_RESET;        // reset the FIFO for Direct Sound A

	/* allocate mixing buffers */
	soundbuffers[0] = malloc(SOUNDBUFFER_LENGTH);
	soundbuffers[1] = malloc(SOUNDBUFFER_LENGTH);
}

void process_soundbuffer()
{
	int i;
	s32 samples_played;
	char foo[100];
	static int fifo_reset_count = 10000;

	s8* buf = soundbuffers[sbuffer];
	sbuffer = !sbuffer;
		
	for(i=0; i < SOUNDBUFFER_LENGTH; i++)
	{
		s32 val = 0;
		int s;
		
		for(s=0; s < SOUNDBUFFER_CHANNELS; s++)
		{
			if(sounds[s].length)
			{
				if((sounds[s].cur + i) < sounds[s].length)
				{
					val += sounds[s].data[sounds[s].cur + i];
					//sounds[s].cur++;
				} else {
					sounds[s].length = 0;
				}
			}
		}
		//val = val / SOUNDBUFFER_CHANNELS;
		val = val >> 1;
		if(val < -128 || val > 128)
		{
			//print("clip!\n");
		}
		buf[i] = val;
	}

	samples_played = REG_TM1D - (0xffff - SOUNDBUFFER_LENGTH);
	//sprintf(foo, "%d\n", samples_played); print(foo);
	for(i=0; i < SOUNDBUFFER_CHANNELS; i++)
	{
		if(sounds[i].length)
		{
			sounds[i].cur += samples_played;
		}
	}
	// make sure Timer 0 is off
	//REG_TM0CNT = 0;

	// make sure DMA channel 1 is turned off
	//REG_DMA1CNT = 0;

	// make sure the FIFO is reset
	if(fifo_reset_count == 10000)
	{
		REG_SOUNDCNT_H |= DSA_FIFO_RESET;   // just set the reset bit and leave the other ones alone
		fifo_reset_count=0;
	}
	fifo_reset_count++;

	// start the timer using the appropriate frequency
	REG_TM0D   = TIMER_INTERVAL;
	REG_TM0CNT = TIMER_ENABLE;

	// start the DMA transfer on channel 1
	REG_DMA1SAD = (u32)(buf);
	REG_DMA1DAD = (u32)REG_FIFO_A;
	REG_DMA1CNT = ENABLE_DMA | START_ON_FIFO_EMPTY | WORD_DMA | DMA_REPEAT;

	// set up timer 1 as a sample length counter
	REG_TM1D = 0xffff - SOUNDBUFFER_LENGTH;
	REG_TM1CNT = TIMER_CASCADE | TIMER_IRQ_ENABLE | TIMER_ENABLE;

	// and enable interrupt used to stop the sample playing when it's finished
	REG_IE |= INT_TM1;
	REG_IME = 1;
}

void play_sound(const SAMPLE* s)
{
	int i;
	for(i=0; i < SOUNDBUFFER_CHANNELS; i++)
	{
		if(sounds[i].length == 0)
		{
			sounds[i].data = s->pData;
			sounds[i].cur = 0;
			sounds[i].length = s->length;
			break;
		}
	}
}

// play_sfx - starts the DMA of a sample and waits for it to complete
// PARAMETERS:  pSample - a pointer to the sample we want to play
// RETURNS:     none
void play_sfx(const s8* data, int length)
{
	// make sure Timer 0 is off
	REG_TM0CNT = 0;

	// make sure DMA channel 1 is turned off
	REG_DMA1CNT = 0;

	// make sure the FIFO is reset
	REG_SOUNDCNT_H |= DSA_FIFO_RESET;   // just set the reset bit and leave the other ones alone

	// start the timer using the appropriate frequency
	REG_TM0D   = TIMER_INTERVAL;
	REG_TM0CNT = TIMER_ENABLE;

	// start the DMA transfer on channel 1
	REG_DMA1SAD = (u32)(data);
	REG_DMA1DAD = (u32)REG_FIFO_A;
	REG_DMA1CNT = ENABLE_DMA | START_ON_FIFO_EMPTY | WORD_DMA | DMA_REPEAT;

	// set up timer 1 as a sample length counter
	REG_TM1D = 0xffff - length;
	REG_TM1CNT = TIMER_CASCADE | TIMER_IRQ_ENABLE | TIMER_ENABLE;

	// and enable interrupt used to stop the sample playing when it's finished
	REG_IE |= INT_TM1;
	REG_IME = 1;
}
