#include "gba.h"
#include "fixed.h"
#include "sound.h"
#include <malloc.h>
#include <string.h>
#include <stdio.h>

#define	INT_TM0		(0x0008)
#define	INT_TM1		(0x0010)

void play_sfx(const s8* data, int length);

/* mixing buffer for sound system */
s8* soundbuffers[4];
u32 sbuffer = 0;

u16 fifo_stalled = 1;

#define SOUNDBUFFER_LENGTH 300

typedef struct
{
	const s8* data;
	u32 length, cur;
	u16 volume; // volume value from 0-16
	s16 panning; // panning value from -8 to 8
	u32 volcache[2]; // volume cache mixing code
} buffer_sound;

#define SOUNDBUFFER_CHANNELS 6
buffer_sound sounds[SOUNDBUFFER_CHANNELS];

// init_sfx_system - sets the registers for enabling the sound hardware and
//                   Direct Sound
// PARAMETERS:  none
// RETURNS:     none
void init_sfx_system(void)
{
	// turn on the sound chip
	REG_SOUNDCNT_X = SND_ENABLED;

	// make sure sound channels 1-4 are turned off
	REG_SOUNDCNT_L = 0;

	// set the direct sound output control register
	REG_SOUNDCNT_H = SND_OUTPUT_RATIO_100 | // 100% sound output
		DSA_OUTPUT_RATIO_100 | // 100% direct sound A output
		DSA_OUTPUT_TO_LEFT |   // output Direct Sound A to left speaker
		DSA_TIMER0 |           // use timer 0 to determine the playback frequency of Direct Sound A
		DSA_FIFO_RESET |        // reset the FIFO for Direct Sound A
		DSB_OUTPUT_RATIO_100 | // 100% direct sound B output
		DSB_OUTPUT_TO_RIGHT |   // output Direct Sound B to right speaker
		DSB_TIMER0 |           // use timer 0 to determine the playback frequency of Direct Sound B
		DSB_FIFO_RESET;        // reset the FIFO for Direct Sound B

	/* allocate mixing buffers */
	soundbuffers[0] = malloc(SOUNDBUFFER_LENGTH);
	soundbuffers[1] = malloc(SOUNDBUFFER_LENGTH);
	soundbuffers[2] = malloc(SOUNDBUFFER_LENGTH);
	soundbuffers[3] = malloc(SOUNDBUFFER_LENGTH);

	REG_TM0D   = TIMER_INTERVAL;
	REG_TM0CNT = TIMER_ENABLE;
	
	/*REG_TM1D = 0xffff - SOUNDBUFFER_LENGTH;
	REG_TM1CNT = TIMER_CASCADE | TIMER_IRQ_ENABLE | TIMER_ENABLE;*/
	
	memset(&sounds[0], 0, sizeof(buffer_sound) * SOUNDBUFFER_CHANNELS);
}

void dump_samples()
{
	/*int i;
	char foo[100];

	for(i=0; i < SOUNDBUFFER_CHANNELS; i++)
	{
		sprintf(foo, "channel: %d\n", i); print(foo);
		sprintf(foo, "\tlen: %d\n", sounds[i].length); print(foo);
		sprintf(foo, "\tcur: %d\n", sounds[i].cur); print(foo);
	}*/
}

void process_soundbuffer()
{
	int i, s;
	int channel;
	s32 samples_played;

	s8* buf[2];

	if(sbuffer)
	{
		sbuffer = 0;
	} else {
		sbuffer = 1;
	}
	buf[0] = soundbuffers[sbuffer];
	buf[1] = soundbuffers[sbuffer + 2];

	//dump_samples();

	/* pre-calculate volume levels for each channel of each sound */
	for(s=0; s < SOUNDBUFFER_CHANNELS; s++)
	{
		sounds[s].volcache[0] = sounds[s].volume * (8 - sounds[s].panning);
		sounds[s].volcache[1] = sounds[s].volume * (8 + sounds[s].panning);
	}

	for(channel=0; channel < 2; channel++)
	{
		for(i=0; i < SOUNDBUFFER_LENGTH; i++)
		{
			s32 val = 0;
		
			for(s=0; s < SOUNDBUFFER_CHANNELS; s++)
			{
				if((sounds[s].cur + i) < sounds[s].length)
				{
					val += ((sounds[s].data[sounds[s].cur + i] * sounds[s].volcache[channel]) >> 8);
				}
			}
			//val = val / SOUNDBUFFER_CHANNELS;
			val = val >> 1;
			if(val < -128 || val > 128)
			{
				//print("clip!\n");
			}
			buf[channel][i] = val;
		}
	}
	samples_played = 265;
	for(i=0; i < SOUNDBUFFER_CHANNELS; i++)
	{
		if(sounds[i].length)
		{
			sounds[i].cur += samples_played;
			if(sounds[i].cur > sounds[i].length)
			{
				sounds[i].cur = 0; sounds[i].length = 0;
			}
		}
	}
	// make sure Timer 0 is off
	REG_TM0CNT = 0;
	REG_TM1CNT = 0;

	// make sure DMA channel 1 is turned off
	REG_DMA1CNT = 0;
	REG_DMA2CNT = 0;

	// make sure the FIFO is reset
	if(fifo_stalled)
	{
		*(volatile u8*)REG_FIFO_A = buf[0][0];
		*(volatile u8*)REG_FIFO_B = buf[1][0];
		REG_SOUNDCNT_H |= (DSA_FIFO_RESET | DSB_FIFO_RESET);   // just set the reset bit and leave the other ones alone
		fifo_stalled = 0;
	}

	// start the timer using the appropriate frequency
	REG_TM0D   = TIMER_INTERVAL;
	REG_TM0CNT = TIMER_ENABLE;

	// start the DMA transfer on channel 1
	REG_DMA1SAD = (u32)(buf[0]);
	REG_DMA1DAD = (u32)REG_FIFO_A;
	REG_DMA1CNT = ENABLE_DMA | START_ON_FIFO_EMPTY | WORD_DMA | DMA_REPEAT;
	
	REG_DMA2SAD = (u32)(buf[1]);
	REG_DMA2DAD = (u32)REG_FIFO_B;
	REG_DMA2CNT = ENABLE_DMA | START_ON_FIFO_EMPTY | WORD_DMA | DMA_REPEAT;

	// set up timer 1 as a sample length counter
	REG_TM1D = 0xffff - (SOUNDBUFFER_LENGTH);
	REG_TM1CNT = TIMER_CASCADE | TIMER_IRQ_ENABLE | TIMER_ENABLE;

	// and enable interrupt used to stop the sample playing when it's finished
	REG_IE |= INT_TM1;
	REG_IME = 1;
}

void play_sound(const SAMPLE* s)
{
	int i;
	for(i=0; i < SOUNDBUFFER_CHANNELS; i++)
	{
		if(sounds[i].length == 0)
		{
			//print("start play!\n");
			sounds[i].data = s->pData;
			sounds[i].cur = 0;
			sounds[i].volume = s->volume;
			sounds[i].panning = s->panning;	
			sounds[i].length = s->length;
			return;
		}
	}
	//print("play failed!\n");
}

// play_sfx - starts the DMA of a sample and waits for it to complete
// PARAMETERS:  pSample - a pointer to the sample we want to play
// RETURNS:     none
void play_sfx(const s8* data, int length)
{
	// make sure Timer 0 is off
	REG_TM0CNT = 0;

	// make sure DMA channel 1 is turned off
	REG_DMA1CNT = 0;

	// make sure the FIFO is reset
	REG_SOUNDCNT_H |= DSA_FIFO_RESET;   // just set the reset bit and leave the other ones alone

	// start the timer using the appropriate frequency
	REG_TM0D   = TIMER_INTERVAL;
	REG_TM0CNT = TIMER_ENABLE;

	// start the DMA transfer on channel 1
	REG_DMA1SAD = (u32)(data);
	REG_DMA1DAD = (u32)REG_FIFO_A;
	REG_DMA1CNT = ENABLE_DMA | START_ON_FIFO_EMPTY | WORD_DMA | DMA_REPEAT;

	// set up timer 1 as a sample length counter
	REG_TM1D = 0xffff - length;
	REG_TM1CNT = TIMER_CASCADE | TIMER_IRQ_ENABLE | TIMER_ENABLE;

	// and enable interrupt used to stop the sample playing when it's finished
	REG_IE |= INT_TM1;
	REG_IME = 1;
}
